// (c) 1999 - 2025 OneSpan North America Inc. All rights reserved.


/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import MSSOrchestration
import LocalAuthentication

// this class is used for usage of external user authentication
class UserAuthenticationViewController: UIViewController {
    private var inputDelegate: UserInputCaller?
    private var isEnrollment = false
    private weak var viewController: UIViewController?
    
    func useExternalPassword(orchestrator: Orchestrator, viewController: UIViewController) {
        self.viewController = viewController
        orchestrator.set(userAuthenticationDelegate: self, for: [.password, .biometric])
    }
    
    private func displayAlert(message: String, viewController: UIViewController) {
        let alertController = UIAlertController(title: "orch_pinpad_text_authentication".localized, message: message, preferredStyle: .alert)
        alertController.addAction(
            UIAlertAction(
                title: "OK",
                style: .default) { [weak self] _ -> Void in
                guard let self else { return }
                guard let textField = alertController.textFields?[0], let viewController = self.viewController else {
                    return
                }
                
                if self.isEnrollment {
                    if let textField2 = alertController.textFields?[1], textField.text != textField2.text {
                        self.displayAlert(message: "orch_pinpad_error_confirmation".localized, viewController: viewController)
                        return
                    }
                }
                
                guard let passwordString = textField.text else {
                    return
                }
                
                let count = passwordString.utf8.count + 1
                let passwordAsCharArray = UnsafeMutablePointer<Int8>.allocate(capacity: count)
                
                passwordString.withCString { baseAddress in
                    passwordAsCharArray.initialize(from: baseAddress, count: count)
                }
                    do {
                        try self.inputDelegate?.userInputDidFinish(charArrayInput: passwordAsCharArray)
                    } catch {
                        print("Happen when this method is used with a biometric authentication")
                    }
            }
        )
        
        alertController.addAction(
            UIAlertAction(
                title: "Cancel",
                style: .cancel) { [weak self] _ -> Void in
                self?.inputDelegate?.userAborted()
            }
        )
        
        alertController.addTextField { [weak self] textField in
            guard let self else { return }
            textField.placeholder = self.isEnrollment ? "orch_pinpad_text_registration".localized : "orch_pinpad_text_authentication".localized
            textField.isSecureTextEntry = true
        }
        
        if isEnrollment {
            alertController.addTextField {textField in
                textField.placeholder = "orch_pinpad_text_registration_confirm".localized
                textField.isSecureTextEntry = true
            }
        }
        viewController.present(alertController, animated: true, completion: nil)
    }
}

extension UserAuthenticationViewController: UserAuthenticationDelegate {
    func orchestrator(_ orchestrator: Orchestrator, userAuthenticationRequiredFor type: UserAuthenticationType, userInputCaller: UserInputCaller, isEnrollment: Bool) {
        inputDelegate = userInputCaller
        self.isEnrollment = isEnrollment
        if let viewController = self.viewController {
            if type == .password {
                displayAlert(message: "", viewController: viewController)
            } else if type == .biometric {
                let context = LAContext()
                var error: NSError?
                guard context.canEvaluatePolicy(.deviceOwnerAuthenticationWithBiometrics, error: &error) else {
                    self.inputDelegate?.userAborted()
                    return
                }
                context.evaluatePolicy(.deviceOwnerAuthenticationWithBiometrics,  localizedReason: "orch_biometric_description".localized) { (success, error) -> () in
                    if success {
                        do {
                            try self.inputDelegate?.userDidFinish()
                        } catch {
                            print("Happen when this method is used with a password authentication")
                        }
                    } else {
                        print(error?.localizedDescription ?? "")
                    }
                }
            }
        }
    }
    
    func orchestrator(_ orchestrator: Orchestrator, didReceiveUserAuthenticationError error: OrchestrationError) {
        if let viewController = self.viewController {
            displayAlert(message: "orch_pinpad_error_weak".localized, viewController: viewController)
        }
    }
}
